---
theme: seriph
title: 基于 LeaferJS 封装的底图渲染与交互源码解析
info: |
  本演讲介绍如何在项目中使用 `mark/` 模块与 `leafer-mark-core/index.ts`，以及该模块通过封装 LeaferJS 的 `class LeaferAnnotate` 的核心设计、插件体系与关键方法。
class: text-center
drawings:
  persist: false
transition: slide-left
mdc: true
---

# 基于 LeaferJS 封装的底图渲染与交互源码解析



---

## 目录

1. 使用场景与目标
2. 快速上手（组件与 Hook）
3. 运行时能力（API 与事件）
4. 源码解读：`LeaferAnnotate` 架构
5. 插件体系与交互模式
6. 数据流与落库
7. 扩展点与最佳实践

---

## 1. 使用场景与目标

- 标注 PDF 页面或图片底图上的题目/区域
- 支持拖拽、复制、吸附、标尺、对齐辅助线
- 统一封装：在 Vue 组件中复用一个 Leafer 标注实例

目标：简单接入、稳定交互、数据可序列化落库

---

## 2. 快速上手

只需要在页面中引入 hook `useLeaferAnnotate`

要点：
- 调用 `createApp` 生成LeaferAnnotate实例，绑定到页面中
- 调用 `loadData(pageUrl, marks)` 加载底图与标注
- 调用 `loadPoints(pointsData)` 加载学员的书写笔迹
```ts
import { onMounted } from 'vue'
import { useLeaferAnnotateSingleton } from '@/mark/leafer-mark-core/useLeaferAnnotate'

const { createApp, loadData, loadPoints } = useLeaferAnnotateSingleton()

onMounted(async () => {
  // 初始化leafer canvas容器
  await createApp({ view: 'leafer-container' })
  // 加载底图与mark
  await loadData('https://example.com/page.png', [])
  // 加载学员笔迹
  loadPoints(pointsData)
})
```

---

## 2. 快速上手（Hook 能力）

`useLeaferAnnotateSingleton` 提供单例式标注实例管理，常用能力：

- createApp(config)
- loadData(pageUrl, marks)
- selectMark / delElement / resetView
- setMarkHover / unsetMarkHover
- onElementSelect / onElementAdd（事件）
- markList（响应式标注列表）

```ts
const {
  createApp,
  loadData,
  markList,
  onElementSelect,
  onElementAdd,
  selectMark,
  delElement,
  resetView
} = useLeaferAnnotateSingleton()

onElementSelect(({ element }) => {
  // 业务联动，如高亮列表项
})

onElementAdd(({ element }) => {
  // 业务联动，如自动选中或滚动到新项
})
```

---



## 4. 源码解读：初始化流程

关键步骤：
- 创建 `App`，容器取自 `config.view`
- 新建 `Frame` 作为页面层 `pageFrame`
- `bindPlugins()` 启用吸附、拖拽、复制、创建矩形、标尺、对齐辅助线
- `changeMode('view')` 默认查看/编辑模式

```ts
async init() {
  this.app = new App({ view: this.config.view, ...DEFAULT_LEAFER_CONFIG })
  this.pageFrame = new Frame({ id: 'pageFrame' })
  this.app.tree?.add(this.pageFrame)
  this.bindPlugins()
  this.changeMode('view')
}
```

---

## 4. 源码解读：加载数据

`loadData(pageUrl, marks)`：
- 清理旧内容与资源缓存
- 通过 `loadImage` 获取底图尺寸，设置 `pageFrame` 宽高
- 添加 `Image` 节点为底图
- `initMarks(marks)` 将服务端标注转成矩形加入 `pageFrame`

```ts
async loadData(pageUrl, marks) {
  this.pageFrame.clear(); Resource.destroy()
  const { url, width, height } = await loadImage(pageUrl, 'bg.png')
  this.pageFrame.width = width
  this.pageFrame.height = height
  this.app.tree?.zoom('fit-width', [32, 12, 12, 32])
  this.pageFrame.add(new Image({ id: 'bg-image', url, width, height }))
  this.initMarks(marks)
}
```

---

## 5. 插件体系与交互模式

`bindPlugins()` 一次性安装：
- Snap（对齐/吸附）
- Ruler（标尺）
- AdsorptionBinding（像素级微调，去小数）
- DropBinding（拖拽）
- CopyRectBinding（Ctrl 拖拽复制）
- CreateRectBinding（画框创建）

交互模式 `changeMode(mode)`：
- view：可缩放/移动，命中子元素
- edit：锁定页面交互，`hitChildren = false`

---

## 6. 数据流与落库

组件侧（`mark/index.vue`）：
- 拉取页列表与对应 `marks`
- `loadData(pageUrl, marks)` 注入底图与标注
- 保存时遍历 `markList`，将像素点位转换为毫米点位并落库

核心转换：

```ts
const dataList = markList.value.map(item => {
  const { top: topPx, bottom: bottomPx } = processRectToMarkPoints(
    item.x as number, item.y as number, item.width as number, item.height as number
  )
  const top = processPxToMmPoint(topPx)
  const bottom = processPxToMmPoint(bottomPx)
  return { ...item.data, top, bottom, topPx, bottomPx }
})
```

---

## 7. 扩展点与最佳实践

- 通过 `marks` 数据结构扩展业务字段（如题型、知识点）
- 在 `onElementAdd / onElementSelect` 中做业务联动（列表、右侧面板）
- 利用 `limit`（宽高/锁定）做页面级约束
- 自定义主题色：`getThemeColor(mark.data)` 控制交互态填充/描边
- 复杂场景下一律使用单例 Hook，避免多实例资源竞争

---

## Q & A

欢迎提问与交流

— 代码位置：
- `mark/leafer-mark-core/core/index.ts`
- `mark/leafer-mark-core/useLeaferAnnotate.ts`
- `mark/index.vue`
